/*
 *  Copyright (C) 2006 Sternest Meanings
 *	contact@sternestmeanings.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package sternestmeanings;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.Random;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This is an implementation of an anagrammer that anagrams strings by
 * querying the Anagram Genius Server at 
 * http://www.anagramgenius.com/server.php
 * 
 * @author Ian Li
 */
public class Anagrammer implements Responder {

	// Maximum time delay (milliseconds) to respond to messages.
	private static final int MAX_DELAY = 1;

	// Maximum length of message that the anagrammer will produce. This is to
	// prevent overloading of the server.
	private static final int MAX_MESSAGE_LENGTH = 150;

	// Maximum length of text that can be processed by the Anagram Genius Server
	private static final int MAX_ANAGRAM_LENGTH = 30;

	private static final String DEFAULT_URL = "http://www.anagramgenius.com/server.php";

	private Pattern responsePattern;

	private static final String PATTERN_START = "<br><span class=\"black-18\">'";

	private static final String PATTERN_END = "'</span></h3>";

	private static final String PATTERN = PATTERN_START + "(.*)" + PATTERN_END;

	public Anagrammer() {
		this.responsePattern = Pattern.compile(PATTERN,
				Pattern.CASE_INSENSITIVE);
	}

	/* (non-Javadoc)
	 * @see sternestmeanings.Responder#anagram(java.lang.String)
	 */
	public String respond(String message) {
		int length = (message.length() > MAX_MESSAGE_LENGTH) ? MAX_MESSAGE_LENGTH
				: message.length();

		StringBuffer response = new StringBuffer();

		int numTokens = (length / MAX_ANAGRAM_LENGTH) + 1;
		for (int i = 0; i < numTokens; i++) {
			int startIndex = i * 30;
			int endIndex = (startIndex + 30 > length) ? length
					: startIndex + 30;

			String statement = message.substring(startIndex, endIndex);

			String anagram = anagramUsingServer(statement);

			if (anagram != null) {
				// Add a space if appending to statement.
				if (response.length() > 0) {
					response.append(" ");
				}

				response.append(anagram);
			}

			pauseRandomly();
		}

		return response.toString();
	}

	private void pauseRandomly() {
		Random random = new Random();
		try {
			Thread.sleep(random.nextInt(MAX_DELAY));
		} catch (InterruptedException e) {
		}
	}

	private String anagramUsingServer(String request) {
		try {
			String query = "?source_text="
					+ URLEncoder.encode(request, "UTF-8");

			// Create a URL for the desired page
			URL url = new URL(DEFAULT_URL + query);

			// Read all the text returned by the server
			BufferedReader in = new BufferedReader(new InputStreamReader(url
					.openStream()));
			StringBuffer buffer = new StringBuffer();
			String str;
			while ((str = in.readLine()) != null) {
				buffer.append(str);
			}
			in.close();

			return parseUsingRegex(buffer.toString());
		} catch (MalformedURLException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return null;
	}

	private String parseUsingRegex(String response) {
		Matcher matcher = responsePattern.matcher(response);
		if (matcher.find()) {
			return matcher.group(1);
		} else {
			return null;
		}
	}

	/**
	 * This is for testing purposes only.
	 */
	public static void main(String[] args) {
		final Responder spider = new Anagrammer();

		System.out.println(spider.respond("this is a test"));
	}
}
