/*
 *  Copyright (C) 2006 Sternest Meanings
 *	contact@sternestmeanings.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package sternestmeanings;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import com.levelonelabs.aim.AIMAdapter;
import com.levelonelabs.aim.AIMBuddy;
import com.levelonelabs.aim.AIMSender;

/**
 * This is the implementation of the AIM bot that responds with anagrams
 * of the user's message.
 * 
 * @author Ian Li
 */
public class GibberAIMBot extends AIMAdapter {
	
	private long delay = 10000;
	
	private String name;
	private AIMSender aim;
	private Map loggers;
	private Map lastMessageTimes;
	
	private Responder responder;
	
	public GibberAIMBot(String name, AIMSender aim, Responder responder) {
		this.name = name;
		this.aim = aim;
		this.responder = responder;
		this.loggers = new HashMap();
		this.lastMessageTimes = new HashMap();
	}
	
	public void setResponder(Responder responder) {
		if (responder != null) {
			this.responder = responder;
		}
	}
	
	public Responder getResponder() {
		return responder;
	}
	
	public void handleMessage(AIMBuddy buddy, String message) {
		Logger logger = getLogger(buddy.getName());
		
		// Log received message
		logger.logReceivedMessage(message);
		
		controlPace(buddy.getName());
		
		String response = responder.respond(message);

		aim.sendMessage(buddy, response);
		
		// Log sent message.
		logger.logSentMessage(response);
	}
	
	public void handleBuddySignOff(AIMBuddy buddy, String request) {
		removeLogger(buddy.getName());
	}
	
	private Logger getLogger(String buddyName) {
		if (loggers.containsKey(buddyName)) {
			return (Logger) loggers.get(buddyName);
		} else {
			Logger logger = new Logger(this.name, buddyName, (new Date()).getTime());
			loggers.put(buddyName, logger);
			return logger;
		}
	}
	
	private void removeLogger(String buddyName) {
		if (loggers.containsKey(buddyName)) {
			loggers.remove(buddyName);
		}
	}
	
	private void controlPace(String buddyName) {
		if (lastMessageTimes.containsKey(buddyName)) {
			Date lastTime = (Date) lastMessageTimes.get(buddyName);
			Date date = new Date();
			
			long timeDifference = date.getTime() - lastTime.getTime();
			
			if (timeDifference < delay) {
				try {
					Thread.sleep(this.delay);
				} catch (InterruptedException e) {
				}
			}
			
			lastMessageTimes.put(buddyName, new Date());
		} else {
			lastMessageTimes.put(buddyName, new Date());
		}
	}
}
