/*
 *  Copyright (C) 2006 Sternest Meanings
 *	contact@sternestmeanings.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package sternestmeanings;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.text.Format;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * This is the class that is responsible for logging conversations.
 * 
 * @author Ian Li
 */
public class Logger {

	private String name;
	private String buddyName; 
	private long startTime;
	private String filename;
	
	public Logger(String name, String buddyName, long startTime) {
		this.name = name;
		this.buddyName = buddyName;
		this.startTime = startTime;
		
		Date date = new Date(startTime);
		
		Format formatter = new SimpleDateFormat(" 'on' yyyy-MM-dd 'at' HH.mm");
		String dateString = formatter.format(date);
		
		this.filename = buddyName + dateString + ".log";
	}
	
	public String getBuddyName() {
		return buddyName;
	}

	public String getName() {
		return name;
	}

	public long getStartTime() {
		return startTime;
	}

	public String getFilename() {
		return this.filename;
	}
	
	public void logReceivedMessage(String message) {
	    try {
	        BufferedWriter out = new BufferedWriter(new FileWriter(this.getFilename(), true));
	        out.write("[" + this.getCurrentTimeString() + "] ");
	        out.write(this.buddyName + ": " + message);
	        out.newLine();
	        out.close();
	    } catch (IOException e) {
	    		e.printStackTrace();
	    }
	}
	
	public void logSentMessage(String message) {
	    try {
	        BufferedWriter out = new BufferedWriter(new FileWriter(this.getFilename(), true));
	        out.write("[" + this.getCurrentTimeString() + "] ");
	        out.write(this.name + ": " + message);
	        out.newLine();
	        out.close();
	    } catch (IOException e) {
	    		e.printStackTrace();
	    }		
	}
	
	/**
	 * Returns a string representing the hour of the day and minutes, e.g. 12:34
	 *
	 * @return a String object representing the current time.
	 */
	private String getCurrentTimeString() {
		Date date = new Date();
		
		Format formatter = new SimpleDateFormat("HH:mm");
		String dateString = formatter.format(date);
		
		return dateString;
	}
	
	/**
	 * This is just for testing purposes.
	 */
	public static void main(String[] args) {
		Logger logger = new Logger("me", "you", (new Date()).getTime());
		
		logger.logReceivedMessage("Received message 1");
		logger.logSentMessage("Sent message 1");
		logger.logReceivedMessage("Received message 2");
		logger.logSentMessage("Sent message 2");
	}
}
