/*
 *  Copyright (C) 2006 Sternest Meanings
 *	contact@sternestmeanings.com
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package sternestmeanings;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Properties;
import java.util.StringTokenizer;

/**
 * This is the main application for the gibber bots using the standard input
 * interface.
 * 
 * @author Ian Li
 */
public class SternestMeaningsStdinApp {
	private Responder responder;
	
	public SternestMeaningsStdinApp(Responder responder) {
		this.responder = responder;
	}

	public void run() {
		try {
			BufferedReader in = new BufferedReader(new InputStreamReader(System.in));	
			PrintWriter out = new PrintWriter(new OutputStreamWriter(System.out));

			String line = "";
			while (line != null) {
				out.print("> ");
				out.flush();
				
				line = in.readLine();
				
				out.println(responder.respond(line));
				out.flush();
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		if (args.length != 1) {
			System.out.println("You must have one argument: properties_file");
		} else {
			final String propertiesFilename = args[0];
			Properties properties = new Properties();

			try {
				properties.load(new FileInputStream(propertiesFilename));
			} catch (IOException e) {
				e.printStackTrace();
			}

			final String responderList = properties
					.getProperty("responderList");
			
			StringTokenizer st = new StringTokenizer(responderList,",",false);
			
			Responder responder;
			if (st.countTokens() == 0) {
				System.out.println("No responders specified. " +
						"Defaulting to DittoResponder.");
				responder = new DittoResponder();
			} else if (st.countTokens() == 1) {
				responder = getResponder(st.nextToken());
			} else {
				AssemblyResponder assembly = new AssemblyResponder();
				while (st.hasMoreTokens()) {
					String token = st.nextToken();
					System.out.println(token);
					assembly.addResponder(getResponder(token));
				}
				responder = assembly;
			}

			SternestMeaningsStdinApp app = new SternestMeaningsStdinApp(responder);
			app.run();
		}
	}
	
	public static Responder getResponder(String responderName) {
		responderName = responderName.trim();

		if (responderName.equals("sternestmeanings")) {
			return new SternestMeaningsResponder();
		} else if (responderName.equals("quote")) {
			return new QuoteResponder();
		} else if (responderName.equals("piglatin")) {
			return new PigLatinResponder();
		} else if (responderName.equals("swedishchef")) {
			return new SwedishChefResponder();
		} else if (responderName.equals("ditto")) {
			return new DittoResponder();
		} else {
			return new DittoResponder();
		}
	}
}
